<xsl:stylesheet
    xmlns:xsl="http://www.w3.org/1999/XSL/Transform"
    xmlns:fn="fn"
    xmlns:pl="http://product-live.com"
    xmlns:xs="http://www.w3.org/2001/XMLSchema"
    version="3.0"
    exclude-result-prefixes="xs fn pl">
    
    <xsl:output method="xml" indent="yes" encoding="UTF-8" cdata-section-elements="" />

    <xsl:param name="sourceSchemaUrl" select="'./TESTS/supplierSchema.xml'" />
    <xsl:param name="retailerSchemaUrl" select="'./TESTS/retailerSchema.xml'" />
    <xsl:param name="retailerRulesUrl" select="'./TESTS/retailerRules.xml'" />
    <xsl:param name="retailerItemsUrl" select="'./TESTS/retailerItems.xml'" />

    <xsl:variable name="sourceSchema" select="doc($sourceSchemaUrl)/Table/Schema" />
    <xsl:variable name="retailerSchema" select="doc($retailerSchemaUrl)/Table/Schema" />
    <xsl:variable name="retailerRules" select="doc($retailerRulesUrl)/Rules-Engine/Rules" />
    <xsl:variable name="retailerItems" select="doc($retailerItemsUrl)/Table/Items" />


    <xsl:variable name="settings">
        <Supplier>
            <Name><xsl:value-of select="doc($sourceSchemaUrl)/Table/@key"/></Name>
            <Classification><xsl:value-of select="$sourceSchema/Classifications/Classification/@key"/></Classification>
        </Supplier>
        <Retailers>
            <Retailer>
                <Name>Retailer</Name>
                <Classification key="CLASSIFICATION" />
                <Screens>
                    <Screen key="ALL" title="Voir toutes les colonnes" />
                </Screens>
                <Conditional-Formattings>
                    <Conditional-Formatting key="CONDITIONNAL-FORMATTING" invalid-status="INVALID" />
                </Conditional-Formattings>
                <xsl:copy-of select="$retailerSchema"/>
                <xsl:copy-of select="$retailerRules"/>
                <xsl:copy-of select="$retailerItems"/>
            </Retailer>
        </Retailers>
    </xsl:variable>
    
    <xsl:template match="/">
        <Generate-Excel>
            <File>
                <File-Name>rules-engine-report-But.xlsx</File-Name>
                <Template-Key>template-report</Template-Key>
                <Sheets>
                    <xsl:for-each select="$settings/Retailers/Retailer">
                        <xsl:variable name="items" select="Items/Item" />
                        <xsl:variable name="countItems" select="count($items)" />
                        <xsl:variable name="rules" select="Rules" />
                        <xsl:variable name="schema" select="Schema" />
                        <xsl:variable name="conditionalFormattings" select="Conditional-Formattings/Conditional-Formatting" />
                        <xsl:variable name="classification" select="Classification/@key" />
                        <xsl:variable name="categories" select="distinct-values($items/Classification[@key=$classification])" />
                        <xsl:variable name="linkedCategories" select="$rules/Classification[@key=$classification]/Rule/Action[@type='SET_CATEGORY']/Map/Entry" /> 
                        <xsl:variable name="screens" select="Screens/Screen/@key/string()" />
                        <xsl:variable name="commonFields" select="distinct-values($schema/Screens/Screen[@key=$screens]/Grid/Common/Section/Column-Field/@key/string())" />
                        <xsl:variable name="linkedCommonFields" select="$rules/Field[@key=$commonFields]" />
                        <xsl:variable name="commonRequiredFields" as="xs:string*">
                            <xsl:variable name="allRequiredFields" as="xs:string*">
                                <xsl:for-each select="$conditionalFormattings">
                                    <xsl:variable name="cfKey" select="@key" />
                                    <xsl:variable name="invalidStatus" select="@invalid-status" />
                                    <xsl:copy-of select="$schema/Conditional-Formattings/Conditional-Formatting[@key=$cfKey]/Statuses/Status[@key=$invalidStatus]/Rules/Common/Field[exists(Rule-Required)]/@key/string()"/>
                                </xsl:for-each>
                            </xsl:variable>
                            <xsl:copy-of select="distinct-values($allRequiredFields)"/>
                        </xsl:variable>
                        <xsl:variable name="linkedCommonRequiredFields" select="$rules/Field[@key=$commonRequiredFields]" />
                        <xsl:variable name="specificFields" select="distinct-values($schema/Screens/Screen[@key=$screens]/Grid/Specific[@classification=$classification and @category=$categories]/Section/Column-Field/@key/string())" />
                        <xsl:variable name="specificRequiredFields" as="xs:string*">
                            <xsl:variable name="allSpecificRequiredFields" as="xs:string*">
                                <xsl:for-each select="$conditionalFormattings">
                                    <xsl:variable name="cfKey" select="@key" />
                                    <xsl:variable name="invalidStatus" select="@invalid-status" />
                                    <xsl:copy-of select="$schema/Conditional-Formattings/Conditional-Formatting[@key=$cfKey]/Statuses/Status[@key=$invalidStatus]/Rules/Specific[@classification=$classification and @category=$categories]/Field[exists(Rule-Required)]/@key/string()"/>
                                </xsl:for-each>
                            </xsl:variable>
                            <xsl:copy-of select="distinct-values($allSpecificRequiredFields)"/>
                        </xsl:variable>

                        <!-- By Categories -->
                        <xsl:variable name="tabCategories">
                            <xsl:for-each select="$categories">
                                <xsl:variable name="position" select="position()+6" />
                                <xsl:variable name="categoryKey" select="." />
                                <xsl:variable name="titleTarget" select="$schema/Classifications/Classification[@key=$classification]/Categories/Category[@key=$categoryKey]/Title" />
                                <Cell-Text line="{$position}" column="3"><xsl:value-of select="$titleTarget"/></Cell-Text>
                                <xsl:choose>
                                    <xsl:when test="exists($linkedCategories[@target=$categoryKey])">
                                        <xsl:variable name="sources" select="$linkedCategories[@target=$categoryKey]/@source" />
                                        <xsl:variable name="validSources" as="xs:string*">
                                            <xsl:for-each select="$sources">
                                                <xsl:variable name="source" select="." />
                                                <xsl:if test="exists($sourceSchema/Classifications/Classification[@key=$settings/Supplier/Classification]/Categories/Category[@key=$source])">
                                                    <xsl:value-of select="$sourceSchema/Classifications/Classification[@key=$settings/Supplier/Classification]/Categories/Category[@key=$source]/Title" />
                                                </xsl:if>
                                            </xsl:for-each>
                                        </xsl:variable>
                                        <xsl:variable name="invalidSources" as="xs:string*">
                                            <xsl:for-each select="$sources">
                                                <xsl:variable name="source" select="." />
                                                <xsl:if test="not(exists($sourceSchema/Classifications/Classification[@key=$settings/Supplier/Classification]/Categories/Category[@key=$source]))">
                                                    <xsl:value-of select="$source" />
                                                </xsl:if>
                                            </xsl:for-each>
                                        </xsl:variable>
                                        <xsl:choose>
                                            <xsl:when test="count($invalidSources) > 0 and count($validSources) = 0">
                                                <Cell-Text line="{$position}" column="4">Error: <xsl:value-of select="string-join($invalidSources,',')"/> does not exist anymore.</Cell-Text>
                                            </xsl:when>
                                            <xsl:when test="count($invalidSources) > 0 and count($validSources) > 0">
                                                <Cell-Text line="{$position}" column="4">Error: <xsl:value-of select="string-join($invalidSources,',')"/> does not exist anymore. Valid: <xsl:value-of select="string-join($validSources,',')"/></Cell-Text>
                                            </xsl:when>
                                            <xsl:otherwise>
                                                <Cell-Text line="{$position}" column="4"><xsl:value-of select="string-join($validSources,',')"/></Cell-Text>
                                            </xsl:otherwise>
                                        </xsl:choose>
                                    </xsl:when>
                                    <xsl:otherwise>
                                        <Cell-Text line="{$position}" column="4">No link</Cell-Text>
                                    </xsl:otherwise>
                                </xsl:choose> 
                                <xsl:variable name="specificFieldsForCategory" select="distinct-values($schema/Screens/Screen[@key=$screens]/Grid/Specific[@classification=$classification and @category=$categoryKey]/Section/Column-Field/@key/string())" />
                                <xsl:variable name="totalFields" select="count($commonFields) + count($specificFieldsForCategory)" />
                                <Cell-Number line="{$position}" column="5"><xsl:value-of select="$totalFields"/></Cell-Number>
                                <xsl:variable name="specificRequiredFieldsForCategory" as="xs:string*">
                                    <xsl:variable name="allrequiredSpecificFields" as="xs:string*">
                                        <xsl:for-each select="$conditionalFormattings">
                                            <xsl:variable name="cfKey" select="@key" />
                                            <xsl:variable name="cfStatus" select="@invalid-status" />
                                            <xsl:copy-of select="$schema/Conditional-Formattings/Conditional-Formatting[@key=$cfKey]/Statuses/Status[@key=$cfStatus]/Rules/Specific[@classification=$classification and @category=$categoryKey]/Field[exists(Rule-Required)]/@key/string()"/>
                                        </xsl:for-each>
                                    </xsl:variable>
                                    <xsl:copy-of select="distinct-values($allrequiredSpecificFields)"/>
                                </xsl:variable>
                                <xsl:variable name="totalRequiredFields" select="count($commonRequiredFields)+count($specificRequiredFieldsForCategory)" />
                                <Cell-Number line="{$position}" column="6"><xsl:value-of select="$totalRequiredFields"/></Cell-Number>
                                <xsl:variable name="linkedSpecificFieldsForCategory" select="$rules/Field[@key=$specificFieldsForCategory]" />
                                <xsl:variable name="totalLinkedFields" select="count($linkedCommonFields) + count($linkedSpecificFieldsForCategory)" /> 
                                <Cell-Number line="{$position}" column="7"><xsl:value-of select="$totalLinkedFields"/></Cell-Number>
                                <xsl:if test="$totalFields > 0">
                                    <xsl:variable name="percentage" select="$totalLinkedFields div $totalFields" /> 
                                    <Cell-Number line="{$position}" column="8"><xsl:value-of select="$percentage"/></Cell-Number>
                                </xsl:if>
                                <xsl:variable name="totalLinkedFieldsRequired" select="count($linkedCommonRequiredFields) + count($rules/Field[@key=$specificRequiredFieldsForCategory])" />
                                <Cell-Number line="{$position}" column="9"><xsl:value-of select="$totalLinkedFieldsRequired"/></Cell-Number>
                                <xsl:if test="$totalRequiredFields > 0">
                                    <Cell-Number line="{$position}" column="10"><xsl:value-of select="$totalLinkedFieldsRequired div $totalRequiredFields"/></Cell-Number>
                                </xsl:if>
                                <xsl:variable name="itemsInCategory" select="$items[Classification[@key=$classification]=$categoryKey]" />
                                <xsl:variable name="ttm">
                                    <xsl:variable name="times">
                                        <xsl:for-each select="$itemsInCategory[exists(@sent)]">
                                            <xsl:variable name="created" select="@created" />
                                            <xsl:variable name="sent" select="@sent" />
                                            <Days><xsl:value-of select="days-from-duration(xs:dateTime($sent) - xs:dateTime($created))" /></Days>
                                        </xsl:for-each>
                                    </xsl:variable>
                                    <xsl:value-of select="round(avg($times/Days),0)"/>
                                </xsl:variable>
                                <xsl:variable name="invalids">
                                    <xsl:choose>
                                        <xsl:when test="$countItems > 0">
                                            <xsl:variable name="invalidItems" as="xs:integer*">
                                                <xsl:for-each select="$itemsInCategory">
                                                    <xsl:variable name="item" select="." />
                                                    <xsl:for-each select="$conditionalFormattings">
                                                        <xsl:variable name="cfKey" select="@key" />
                                                        <xsl:variable name="cfStatus" select="@invalid-status" />
                                                        <xsl:if test="$item[Conditional-Formatting[@key=$cfKey]=$cfStatus]">
                                                            <xsl:value-of select="1"/>
                                                        </xsl:if>
                                                    </xsl:for-each>
                                                </xsl:for-each>
                                            </xsl:variable>
                                            <xsl:value-of select="count($invalidItems)"/>
                                        </xsl:when>
                                        <xsl:otherwise>-</xsl:otherwise>
                                    </xsl:choose>
                                </xsl:variable>
                                <xsl:variable name="countAverageFilledFields">
                                    <xsl:choose>
                                        <xsl:when test="$countItems > 0">
                                            <xsl:variable name="counts">
                                                <xsl:for-each select="$itemsInCategory[exists(Field)]">
                                                    <Count><xsl:value-of select="count(Field)"/></Count>
                                                </xsl:for-each>
                                            </xsl:variable>
                                            <xsl:value-of select="avg($counts/Count)"/>
                                        </xsl:when>
                                        <xsl:otherwise><xsl:value-of select="0"/></xsl:otherwise>
                                    </xsl:choose>
                                </xsl:variable>
                                <xsl:variable name="completeness">
                                    <xsl:choose>
                                        <xsl:when test="$countItems > 0 and number($countAverageFilledFields) > 0">
                                            <xsl:value-of select="$countAverageFilledFields div (count($commonFields) + count($specificFieldsForCategory))"/>
                                        </xsl:when>
                                        <xsl:otherwise>-</xsl:otherwise>
                                    </xsl:choose>
                                </xsl:variable>                    
                                <Cell-Number line="{$position}" column="11"><xsl:value-of select="count($itemsInCategory)"/></Cell-Number>                                
                                <Cell-Number line="{$position}" column="12"><xsl:value-of select="$invalids"/></Cell-Number>                                
                                <Cell-Number line="{$position}" column="13"><xsl:value-of select="$completeness"/></Cell-Number>                                
                                <Cell-Number line="{$position}" column="14"><xsl:value-of select="$ttm"/></Cell-Number>
                                <Cell-Number line="{$position}" column="15"><xsl:value-of select="count($itemsInCategory[exists(@sent)])"/></Cell-Number>
                                <Cell-Number line="{$position}" column="16"><xsl:value-of select="count($itemsInCategory[@updated > @sent])"/></Cell-Number>            
                                <Cell-Number line="{$position}" column="17"><xsl:value-of select="count($itemsInCategory[not(exists(@sent))])"/></Cell-Number>
                            </xsl:for-each>
                        </xsl:variable>

                        <!-- By Properties -->
                        <xsl:variable name="tabProperties">
                            <xsl:variable name="lines">
                                <xsl:for-each select="Screens/Screen">
                                    <xsl:variable name="screenKey" select="@key" />
                                    <xsl:variable name="screenTitle" select="@title" />
                                    <xsl:for-each select="$commonFields">
                                        <xsl:variable name="fieldKey" select="." />
                                       <xsl:variable name="field" select="$schema/*/*[@key=$fieldKey]"/>
                                        <xsl:variable name="title" select="$field/Title" />
                                        <xsl:variable name="type" select="$field/@type" />
                                        <xsl:variable name="isRequired" select="if (exists($commonRequiredFields[.=$fieldKey])) then 'Yes' else 'No'" />
                                        <xsl:variable name="hasRule" select="if (exists($rules[Field/@key=$fieldKey])) then 'Yes' else 'No'" />
                                        <xsl:variable name="hasIgnoredRule" select="if (exists($rules/*[@key=$fieldKey and @skip='true'])) then 'Yes' else 'No'" />
                                        <xsl:variable name="ruleStatus">
                                            <!-- Check that fields used as @source still exists -->
                                            <xsl:if test="some $f in $rules/Field[@key=$fieldKey]/Rule//@source[not(../name()='Entry')] satisfies not(exists($sourceSchema/*/*[@key=$f]))">
                                                <Error>A field used as source does not exist anymore.</Error>
                                            </xsl:if>
                                            <!-- Check for templates -->
                                            <xsl:if test="exists($rules/Field[@key=$fieldKey]/Rule/Action/Template)">
                                                <xsl:variable name="templateSources" as="xs:string*">
                                                    <xsl:for-each select="tokenize($rules/Field[@key=$fieldKey]/Rule[1]/Action/Template,'\{\{source\(&quot;')">
                                                        <xsl:if test="normalize-space(tokenize(.,'&quot;\)\}\}')[1]) != ''">
                                                            <xsl:value-of select="normalize-space(tokenize(.,'&quot;\)\}\}')[1])"/>
                                                        </xsl:if>                                                
                                                    </xsl:for-each>
                                                </xsl:variable>
                                                <xsl:if test="some $f in $templateSources satisfies not(exists($sourceSchema/*/*[@key=$f]))">
                                                    <Error>A field used as source on a template does not exist anymore.</Error>
                                                </xsl:if>
                                            </xsl:if>
                                            <!-- For single select fields, check that all mapped options still exist in the source -->
                                            <xsl:if test="$type = ('SINGLE-SELECT','MULTIPLE-SELECT')">
                                                <xsl:variable name="sourceFieldKey" select="$rules/Field[@key=$fieldKey]/Rule/Action/Map/@source" />
                                                <xsl:variable name="sourceOptionKeys" select="$rules/Field[@key=$fieldKey]/Rule/Action/Map/Entry/@source" />
                                                <xsl:if test="some $o in $sourceOptionKeys satisfies not(exists($sourceSchema/Fields/Field[@key=$sourceFieldKey]/Options/Option[@key=$o]))">
                                                    <Error>An option used as source does not exist anymore.</Error>
                                                </xsl:if>
                                            </xsl:if>
                                        </xsl:variable>
                                        <xsl:variable name="sources" as="xs:string*">
                                            <xsl:if test="$hasRule = 'Yes'">
                                                <xsl:for-each select="$rules/Field[@key=$fieldKey]/Rule">
                                                    <xsl:choose>
                                                        <xsl:when test="Action/@type = ('SET_TEXT','SET_NUMBER','SET_IMAGE','SET_ATTACHMENT')">
                                                            <xsl:choose>
                                                                <xsl:when test="exists(Action/Copy/@source)">
                                                                    <xsl:variable name="source" select="Action/Copy/@source" />
                                                                    <xsl:choose>
                                                                        <xsl:when test="exists($sourceSchema/Fields/Field[@key=$source])">
                                                                            <xsl:value-of select="$sourceSchema/Fields/Field[@key=$source]/Title"/>
                                                                        </xsl:when>
                                                                        <xsl:when test="exists($sourceSchema/Classifications/Classification[@key=$source])">
                                                                            <xsl:value-of select="$sourceSchema/Classifications/Classification[@key=$source]/Title"/>
                                                                        </xsl:when>
                                                                        <xsl:when test="exists($sourceSchema/Identifiers/Identifier[@key=$source])">
                                                                            <xsl:value-of select="$sourceSchema/Identifiers/Identifier[@key=$source]/Title"/>
                                                                        </xsl:when>
                                                                    </xsl:choose>                                                                    
                                                                </xsl:when>
                                                                <xsl:otherwise>
                                                                    <xsl:variable name="value" select="normalize-space(Action/Value)" />
                                                                    <xsl:value-of select="concat('Default value: ',$value)"/>
                                                                </xsl:otherwise>
                                                            </xsl:choose>
                                                        </xsl:when>
                                                        <xsl:when test="Action/@type = ('SET_OPTION','SET_OPTIONS')">
                                                            <xsl:choose>
                                                                <xsl:when test="exists(Action/Map/@source)">
                                                                    <xsl:variable name="source" select="Action/Map/@source" />
                                                                    <xsl:choose>
                                                                        <xsl:when test="exists($sourceSchema/Fields/Field[@key=$source])">
                                                                            <xsl:value-of select="$sourceSchema/Fields/Field[@key=$source]/Title"/>
                                                                        </xsl:when>
                                                                        <xsl:when test="exists($sourceSchema/Classifications/Classification[@key=$source])">
                                                                            <xsl:value-of select="$sourceSchema/Classifications/Classification[@key=$source]/Title"/>
                                                                        </xsl:when>
                                                                        <xsl:when test="exists($sourceSchema/Identifiers/Identifier[@key=$source])">
                                                                            <xsl:value-of select="$sourceSchema/Identifiers/Identifier[@key=$source]/Title"/>
                                                                        </xsl:when>
                                                                    </xsl:choose>
                                                                </xsl:when>
                                                                <xsl:otherwise>
                                                                    <!-- Default value -->
                                                                    <xsl:variable name="value" as="xs:string*">
                                                                        <xsl:for-each select="Action/Value">
                                                                            <xsl:value-of select="normalize-space(.)"/>
                                                                        </xsl:for-each>
                                                                    </xsl:variable>
                                                                    <xsl:value-of select="concat('Default value: ',string-join($schema/Fields/Field[@key=$fieldKey]/Options/Option[@key=$value]/Title,','))"/>
                                                                </xsl:otherwise>
                                                            </xsl:choose>                                                            
                                                        </xsl:when>
                                                        <xsl:otherwise>Complex</xsl:otherwise>
                                                    </xsl:choose>
                                                </xsl:for-each>
                                            </xsl:if>
                                        </xsl:variable>
                                        <xsl:variable name="completion" select="count($items[Field[@key=$fieldKey]]) div $countItems" />
                                        <Line>
                                            <Cell-Text column="3"><xsl:value-of select="$screenTitle"/></Cell-Text>
                                            <Cell-Text column="4">-</Cell-Text>
                                            <Cell-Text column="5">-</Cell-Text>
                                            <Cell-Text column="6"><xsl:value-of select="$title"/></Cell-Text>
                                            <Cell-Number column="7"><xsl:value-of select="$completion"/></Cell-Number>
                                            <Cell-Text column="8"><xsl:value-of select="$type"/></Cell-Text>
                                            <Cell-Text column="9"><xsl:value-of select="$isRequired"/></Cell-Text>
                                            <xsl:choose>
                                                <xsl:when test="$hasIgnoredRule = 'Yes'">
                                                    <Cell-Text column="10">Skipped</Cell-Text>
                                                </xsl:when>
                                                <xsl:when test="$hasRule = 'Yes'">
                                                    <xsl:choose>
                                                        <xsl:when test="count($sources) = 0">
                                                            <Cell-Text column="10">Broken</Cell-Text>
                                                            <Cell-Text column="12">No rule has been saved</Cell-Text>
                                                        </xsl:when>
                                                        <xsl:when test="exists($ruleStatus/Error)">
                                                            <Cell-Text column="10">Broken</Cell-Text>
                                                            <Cell-Text column="11"><xsl:value-of select="string-join($sources,',')"/></Cell-Text>
                                                            <Cell-Text column="12"><xsl:value-of select="string-join($ruleStatus/Error,',')"/></Cell-Text>
                                                        </xsl:when>
                                                        <xsl:otherwise>
                                                            <Cell-Text column="10">Valid</Cell-Text>
                                                            <Cell-Text column="11"><xsl:value-of select="string-join($sources,',')"/></Cell-Text>
                                                        </xsl:otherwise>
                                                    </xsl:choose>
                                                </xsl:when>
                                                <xsl:otherwise>
                                                    <Cell-Text column="10">No</Cell-Text>
                                                </xsl:otherwise>
                                            </xsl:choose> 
                                        </Line>
                                    </xsl:for-each>
                                    <xsl:variable name="sourceClassificationKey" select="$rules/Classification[@key=$classification]/Rule/Action/Map/@source" />
                                    <xsl:for-each select="$schema/Screens/Screen[@key=$screenKey]/Grid/Specific[@category=$categories]">
                                        <xsl:variable name="categoryKey" select="@category" />
                                        <xsl:variable name="categoryTitle" select="$schema/Classifications/Classification[@key=$classification]/Categories/Category[@key=$categoryKey]/Title" />
                                        <xsl:variable name="sourceCategoryKeys" select="$linkedCategories[@target=$categoryKey]/@source" />
                                        <xsl:variable name="sourceCategoryTitles" select="$sourceSchema/Classifications/Classification[@key=$sourceClassificationKey]/Categories/Category[@key=$sourceCategoryKeys]/Title" />
                                        <xsl:for-each select="Section/Column-Field/@key/string()">
                                            <xsl:variable name="fieldKey" select="." />
                                           <xsl:variable name="field" select="$schema/*/*[@key=$fieldKey]"/>
                                            <xsl:variable name="title" select="$field/Title" />
                                            <xsl:variable name="type" select="$field/@type" />
                                            <xsl:variable name="specificRequiredFieldsForCategory" as="xs:string*">
                                                <xsl:variable name="allrequiredSpecificFields" as="xs:string*">
                                                    <xsl:for-each select="$conditionalFormattings">
                                                        <xsl:variable name="cfKey" select="@key" />
                                                        <xsl:variable name="cfStatus" select="@invalid-status" />
                                                        <xsl:copy-of select="$schema/Conditional-Formattings/Conditional-Formatting[@key=$cfKey]/Statuses/Status[@key=$cfStatus]/Rules/Specific[@classification=$classification and @category=$categoryKey]/Field[exists(Rule-Required)]/@key/string()"/>
                                                    </xsl:for-each>
                                                </xsl:variable>
                                                <xsl:copy-of select="distinct-values($allrequiredSpecificFields)"/>
                                            </xsl:variable>
                                            <xsl:variable name="isRequired" select="if (exists($specificRequiredFieldsForCategory[.=$fieldKey])) then 'Yes' else 'No'" />
                                            <xsl:variable name="hasRule" select="if (exists($rules[Field/@key=$fieldKey])) then 'Yes' else 'No'" />
                                            <xsl:variable name="hasIgnoredRule" select="if (exists($rules/*[@key=$fieldKey and @skip='true'])) then 'Yes' else 'No'" />
                                            <xsl:variable name="ruleStatus">
                                                <!-- Check that fields used as @source still exists -->
                                                <xsl:if test="some $f in $rules/Field[@key=$fieldKey]/Rule//@source[not(../name()='Entry')] satisfies not(exists($sourceSchema/*/*[@key=$f]))">
                                                    <Error>A field used as source does not exist anymore.</Error>
                                                </xsl:if>
                                                <!-- Check for templates -->
                                                <xsl:if test="exists($rules/Field[@key=$fieldKey]/Rule/Action/Template)">
                                                    <xsl:variable name="templateSources" as="xs:string*">
                                                        <xsl:for-each select="tokenize($rules/Field[@key=$fieldKey]/Rule/Action/Template,'\{\{source\(&quot;')">
                                                            <xsl:if test="normalize-space(tokenize(.,'&quot;\)\}\}')[1]) != ''">
                                                                <xsl:value-of select="normalize-space(tokenize(.,'&quot;\)\}\}')[1])"/>
                                                            </xsl:if>                                                
                                                        </xsl:for-each>
                                                    </xsl:variable>
                                                    <xsl:if test="some $f in $templateSources satisfies not(exists($sourceSchema/*/*[@key=$f]))">
                                                        <Error>A field used as source on a template does not exist anymore.</Error>
                                                    </xsl:if>
                                                </xsl:if>
                                                <!-- For single select fields, check that all mapped options still exist in the source -->
                                                <xsl:if test="$type = ('SINGLE-SELECT','MULTIPLE-SELECT')">
                                                    <xsl:variable name="sourceFieldKey" select="$rules/Field[@key=$fieldKey]/Rule/Action/Map/@source" />
                                                    <xsl:variable name="sourceOptionKeys" select="$rules/Field[@key=$fieldKey]/Rule/Action/Map/Entry/@source" />                                    
                                                    <xsl:if test="some $o in $sourceOptionKeys satisfies not(exists($sourceSchema/Fields/Field[@key=$sourceFieldKey]/Options/Option[@key=$o]))">
                                                        <Error>An option used as source does not exist anymore.</Error>
                                                    </xsl:if>
                                                </xsl:if>
                                            </xsl:variable>
                                            <xsl:variable name="sources" as="xs:string*">
                                                <xsl:if test="$hasRule = 'Yes'">
                                                    <xsl:for-each select="$rules/Field[@key=$fieldKey]/Rule">
                                                        <xsl:choose>
                                                            <xsl:when test="Action/@type = ('SET_TEXT','SET_NUMBER','SET_IMAGE','SET_ATTACHMENT')">
                                                                <xsl:choose>
                                                                    <xsl:when test="exists(Action/Copy/@source)">
                                                                        <xsl:variable name="source" select="Action/Copy/@source" />
                                                                        <xsl:choose>
                                                                            <xsl:when test="exists($sourceSchema/Fields/Field[@key=$source])">
                                                                                <xsl:value-of select="$sourceSchema/Fields/Field[@key=$source]/Title"/>
                                                                            </xsl:when>
                                                                            <xsl:when test="exists($sourceSchema/Classifications/Classification[@key=$source])">
                                                                                <xsl:value-of select="$sourceSchema/Classifications/Classification[@key=$source]/Title"/>
                                                                            </xsl:when>
                                                                            <xsl:when test="exists($sourceSchema/Identifiers/Identifier[@key=$source])">
                                                                                <xsl:value-of select="$sourceSchema/Identifiers/Identifier[@key=$source]/Title"/>
                                                                            </xsl:when>
                                                                        </xsl:choose>                                                                    
                                                                    </xsl:when>
                                                                    <xsl:otherwise>
                                                                        <xsl:variable name="value" select="normalize-space(Action/Value)" />
                                                                        <xsl:value-of select="concat('Default value: ',$value)"/>
                                                                    </xsl:otherwise>
                                                                </xsl:choose>
                                                            </xsl:when>
                                                            <xsl:when test="Action/@type = ('SET_OPTION','SET_OPTIONS')">
                                                                <xsl:choose>
                                                                    <xsl:when test="exists(Action/Map/@source)">
                                                                        <xsl:variable name="source" select="Action/Map/@source" />
                                                                        <xsl:choose>
                                                                            <xsl:when test="exists($sourceSchema/Fields/Field[@key=$source])">
                                                                                <xsl:value-of select="$sourceSchema/Fields/Field[@key=$source]/Title"/>
                                                                            </xsl:when>
                                                                            <xsl:when test="exists($sourceSchema/Classifications/Classification[@key=$source])">
                                                                                <xsl:value-of select="$sourceSchema/Classifications/Classification[@key=$source]/Title"/>
                                                                            </xsl:when>
                                                                            <xsl:when test="exists($sourceSchema/Identifiers/Identifier[@key=$source])">
                                                                                <xsl:value-of select="$sourceSchema/Identifiers/Identifier[@key=$source]/Title"/>
                                                                            </xsl:when>
                                                                        </xsl:choose>
                                                                    </xsl:when>
                                                                    <xsl:otherwise>
                                                                        <!-- Default value -->
                                                                        <xsl:variable name="value" as="xs:string*">
                                                                            <xsl:for-each select="Action/Value">
                                                                                <xsl:value-of select="normalize-space(.)"/>
                                                                            </xsl:for-each>
                                                                        </xsl:variable>
                                                                        <xsl:value-of select="concat('Default value: ',string-join($schema/Fields/Field[@key=$fieldKey]/Options/Option[@key=$value]/Title,','))"/>
                                                                    </xsl:otherwise>
                                                                </xsl:choose>                                                            
                                                            </xsl:when>
                                                            <xsl:otherwise>Complex</xsl:otherwise>
                                                        </xsl:choose>
                                                    </xsl:for-each>
                                                </xsl:if>
                                            </xsl:variable>                                    
                                            <xsl:variable name="completion" select="count($items[Classification[@key=$classification]=$categoryKey and Field[@key=$fieldKey]]) div $countItems" />
                                            <Line>
                                                <Cell-Text column="3"><xsl:value-of select="$screenTitle"/></Cell-Text>
                                                <Cell-Text column="4"><xsl:value-of select="$categoryTitle"/></Cell-Text>
                                                <xsl:choose>
                                                    <xsl:when test="count($sourceCategoryTitles) > 0">
                                                        <Cell-Text column="5"><xsl:value-of select="string-join($sourceCategoryTitles,',')"/></Cell-Text>
                                                    </xsl:when>
                                                    <xsl:otherwise>
                                                        <Cell-Text column="5">-</Cell-Text>
                                                    </xsl:otherwise>
                                                </xsl:choose>
                                                <Cell-Text column="6"><xsl:value-of select="$title"/></Cell-Text>
                                                <Cell-Number column="7"><xsl:value-of select="$completion"/></Cell-Number>
                                                <Cell-Text column="8"><xsl:value-of select="$type"/></Cell-Text>
                                                <Cell-Text column="9"><xsl:value-of select="$isRequired"/></Cell-Text>
                                                <xsl:choose>
                                                    <xsl:when test="$hasRule = 'Yes'">
                                                        <xsl:choose>
                                                            <xsl:when test="count($sources) = 0">
                                                                <Cell-Text column="10">Broken</Cell-Text>
                                                                <Cell-Text column="12">No rule has been saved</Cell-Text>
                                                            </xsl:when>
                                                            <xsl:when test="exists($ruleStatus/Error)">
                                                                <Cell-Text column="10">Broken</Cell-Text>
                                                                <Cell-Text column="11"><xsl:value-of select="string-join($sources,',')"/></Cell-Text>
                                                                <Cell-Text column="12"><xsl:value-of select="string-join($ruleStatus/Error,',')"/></Cell-Text>
                                                            </xsl:when>
                                                            <xsl:otherwise>
                                                                <Cell-Text column="10">Valid</Cell-Text>
                                                                <Cell-Text column="11"><xsl:value-of select="string-join($sources,',')"/></Cell-Text>
                                                            </xsl:otherwise>
                                                        </xsl:choose>
                                                    </xsl:when>
                                                    <xsl:otherwise>
                                                        <Cell-Text column="10">No</Cell-Text>
                                                    </xsl:otherwise>
                                                </xsl:choose>
                                            </Line>
                                        </xsl:for-each>
                                    </xsl:for-each>
                                </xsl:for-each> 
                            </xsl:variable>
                            <xsl:for-each select="$lines/Line">
                                <xsl:variable name="position" select="position()+6" />
                                <xsl:for-each select="Cell-Text">
                                    <Cell-Text line="{$position}" column="{@column}"><xsl:value-of select="."/></Cell-Text>
                                </xsl:for-each>
                                <xsl:for-each select="Cell-Number">
                                    <Cell-Number line="{$position}" column="{@column}"><xsl:value-of select="."/></Cell-Number>
                                </xsl:for-each>
                            </xsl:for-each>
                        </xsl:variable>

                        <!-- By options -->
                        <xsl:variable name="tabOptions">
                            <xsl:variable name="lines">
                                <xsl:variable name="fields" select="$schema/Fields/Field[@key=($commonFields,$specificFields) and @type=('SINGLE-SELECT','MULTIPLE-SELECT','MULTIPLE-SELECT-QUANTIFIED','MULTIPLE-SELECT-QUANTIFIED-WITH-COMMENTS')]" />
                                <xsl:for-each select="$fields">
                                    <xsl:variable name="fieldKey" select="@key" />
                                    <xsl:variable name="fieldTitle" select="Title" />
                                    <xsl:variable name="sourceFieldKey" select="$rules/Field[@key=$fieldKey]/Rule/Action/Map/@source" />
                                    <xsl:variable name="sourceField">
                                        <xsl:choose>
                                            <xsl:when test="exists($sourceSchema/Fields/Field[@key=$sourceFieldKey])">
                                                <xsl:copy-of select="$sourceSchema/Fields/Field[@key=$sourceFieldKey]"/>
                                            </xsl:when>
                                            <xsl:when test="exists($sourceSchema/Classifications/Classification[@key=$sourceFieldKey])">
                                                <!-- A trick to avoid duplicating to much lines of code below -->
                                                <Field key="{$sourceFieldKey}">
                                                    <Title><xsl:value-of select="$sourceSchema/Classifications/Classification[@key=$sourceFieldKey]/Title"/></Title>
                                                    <xsl:for-each select="$sourceSchema/Classifications/Classification[@key=$sourceFieldKey]/Categories/Category">
                                                        <Option key="{@key}">
                                                            <Title><xsl:value-of select="Title"/></Title>
                                                        </Option>
                                                    </xsl:for-each>
                                                </Field>
                                            </xsl:when>
                                        </xsl:choose>
                                    </xsl:variable>
                                    <xsl:variable name="sourceFieldTitle" select="$sourceField/Field/Title" />
                                    <xsl:variable name="entries" select="$rules/Field[@key=$fieldKey]/Rule/Action/Map/Entry" />
                                    <xsl:for-each select="Options/Option">
                                        <xsl:variable name="optionKey" select="@key" />
                                        <Line>
                                            <Cell-Text column="3"><xsl:value-of select="$fieldTitle"/></Cell-Text>
                                            <xsl:if test="$sourceFieldTitle != ''">
                                                <Cell-Text column="4"><xsl:value-of select="$sourceFieldTitle"/></Cell-Text>
                                            </xsl:if>                                    
                                            <Cell-Text column="5"><xsl:value-of select="Title"/></Cell-Text>
                                            <xsl:choose>
                                                <xsl:when test="$entries[@target=$optionKey and @skip='true']">
                                                    <Cell-Text column="7">Skipped</Cell-Text>
                                                </xsl:when>
                                                <xsl:when test="exists($entries[@target=$optionKey])">
                                                    <xsl:variable name="sourceOptionKeys" select="$entries[@target=$optionKey]/@source/string()" />
                                                    <xsl:variable name="validSources" as="xs:string*">
                                                        <xsl:for-each select="$sourceOptionKeys">
                                                            <xsl:variable name="optionKey" select="." />
                                                            <xsl:if test="exists($sourceField/Field/Options/Option[@key=$optionKey])">
                                                                <xsl:value-of select="$sourceField/Field/Options/Option[@key=$optionKey]/Title"/>
                                                            </xsl:if>
                                                        </xsl:for-each>
                                                    </xsl:variable>
                                                    <xsl:variable name="invalidSources" as="xs:string*">
                                                        <xsl:for-each select="$sourceOptionKeys">
                                                            <xsl:variable name="optionKey" select="." />
                                                            <xsl:if test="not(exists($sourceField/Field/Options/Option[@key=$optionKey]))">
                                                                <xsl:value-of select="$optionKey"/>
                                                            </xsl:if>
                                                        </xsl:for-each>
                                                    </xsl:variable>
                                                    <xsl:choose>
                                                        <xsl:when test="count($invalidSources) = 0">
                                                            <Cell-Text column="6"><xsl:value-of select="string-join($sourceField/Field/Options/Option[@key=$sourceOptionKeys]/Title,',')"/></Cell-Text>
                                                            <Cell-Text column="7">Valid</Cell-Text>
                                                        </xsl:when>
                                                        <xsl:when test="count($invalidSources) > 0 and count($validSources) > 0">
                                                            <Cell-Text column="8">Option with key: "<xsl:value-of select="string-join($invalidSources,',')"/>" does not exist in the source anymore.</Cell-Text>
                                                            <Cell-Text column="6"><xsl:value-of select="string-join($validSources,',')"/></Cell-Text>
                                                            <Cell-Text column="7">Broken</Cell-Text>
                                                        </xsl:when>
                                                        <xsl:when test="count($invalidSources) > 0 and count($validSources) = 0">
                                                            <Cell-Text column="8">Option with key: "<xsl:value-of select="string-join($invalidSources,',')"/>" does not exist in the source anymore.</Cell-Text>
                                                            <Cell-Text column="7">Broken</Cell-Text>
                                                        </xsl:when>
                                                    </xsl:choose>
                                                </xsl:when>
                                                <xsl:otherwise>
                                                    <Cell-Text column="7">No</Cell-Text>
                                                </xsl:otherwise>
                                            </xsl:choose>
                                        </Line>
                                    </xsl:for-each>
                                </xsl:for-each>
                            </xsl:variable>
                            <xsl:for-each select="$lines/Line">
                                <xsl:variable name="position" select="position()+6" />
                                <xsl:for-each select="Cell-Text">
                                    <Cell-Text line="{$position}" column="{@column}"><xsl:value-of select="."/></Cell-Text>
                                </xsl:for-each>
                            </xsl:for-each>
                        </xsl:variable>

                        <!-- Main -->
                        <xsl:variable name="tabMain">
                            <xsl:variable name="position" select="position()+6" />
                            <Cell-Text line="{$position}" column="3"><xsl:value-of select="Name"/></Cell-Text>
                            <Cell-Number line="{$position}" column="4"><xsl:value-of select="count($categories)"/></Cell-Number>
                            <Cell-Number line="{$position}" column="5"><xsl:value-of select="count($rules/Classification[@key=$classification]/Rule/Action[@type='SET_CATEGORY']/Map/Entry)"/></Cell-Number>
                            <Cell-Number line="{$position}" column="6"><xsl:value-of select="count($commonFields)"/></Cell-Number>
                            <Cell-Number line="{$position}" column="7"><xsl:value-of select="count($commonRequiredFields)"/></Cell-Number>
                            <Cell-Number line="{$position}" column="8"><xsl:value-of select="count($specificFields)"/></Cell-Number>
                            <Cell-Number line="{$position}" column="9"><xsl:value-of select="count($specificRequiredFields)"/></Cell-Number>
                            <xsl:variable name="countFields" select="count($commonFields) + count($specificFields)" />
                            <Cell-Number line="{$position}" column="10"><xsl:value-of select="$countFields"/></Cell-Number>
                            <xsl:variable name="countRequiredFields" select="count($commonRequiredFields) + count($specificRequiredFields)" />
                            <Cell-Number line="{$position}" column="11"><xsl:value-of select="$countRequiredFields"/></Cell-Number>
                            <xsl:variable name="countLinkedFields" select="count($rules/Field[@key=($commonFields,$specificFields)])" />
                            <Cell-Number line="{$position}" column="12"><xsl:value-of select="$countLinkedFields"/></Cell-Number>
                            <xsl:variable name="countRequiredLinkedFields" select="count($rules/Field[@key=($commonRequiredFields,$specificRequiredFields)])" />
                            <Cell-Number line="{$position}" column="13"><xsl:value-of select="$countRequiredLinkedFields"/></Cell-Number>
                            <xsl:if test="$countFields > 0">
                                <Cell-Number line="{$position}" column="14"><xsl:value-of select="$countLinkedFields div $countFields"/></Cell-Number>
                            </xsl:if>
                            <xsl:if test="$countRequiredFields > 0">
                                <Cell-Number line="{$position}" column="15"><xsl:value-of select="$countRequiredLinkedFields div $countRequiredFields"/></Cell-Number>
                            </xsl:if>   
                            <xsl:variable name="countCells">
                                <xsl:variable name="countLinkedCommonFields" select="count($rules/Field[@key=$commonFields])" />
                                <xsl:variable name="cellsByCategories" as="xs:integer*">
                                    <xsl:for-each select="distinct-values($rules/Classification/Rule/Action[@type='SET_CATEGORY']/Map/Entry/@target)">
                                        <xsl:variable name="categoryKey" select="." />
                                        <xsl:variable name="specificFields" select="$schema/Screens/Screen[@key=$screens]/Grid/Specific[@classification=$classification and @category=$categoryKey]/Section/Column-Field/@key/string()" />
                                        <xsl:value-of select="count($items[Classification[@key=$classification]=$categoryKey]) * count($rules/Field[@key=$specificFields])"/>
                                    </xsl:for-each>
                                </xsl:variable>       
                                <xsl:value-of select="count($items) * $countLinkedCommonFields + sum($cellsByCategories)"/>
                            </xsl:variable>
                            <Cell-Number line="{$position}" column="16"><xsl:value-of select="$countCells"/></Cell-Number>                            
                            <Cell-Number line="{$position}" column="18"><xsl:value-of select="count($items)"/></Cell-Number>                            
                            <Cell-Number line="{$position}" column="19"><xsl:value-of select="count($items[not(Classification[@key=$classification])])"/></Cell-Number>                            
                            <Cell-Number line="{$position}" column="20"><xsl:value-of select="count($items[Conditional-Formatting=$conditionalFormattings/@invalid-status])"/></Cell-Number>
                            <!-- Completeness -->
                            <xsl:variable name="completeness">
                                <xsl:variable name="countCompletedFields" select="count($items[Classification[@key=$classification]=$categories]/Field)" />
                                <xsl:variable name="countExpectedFields">
                                    <xsl:variable name="expectedFields">
                                        <xsl:variable name="specifics" select="$schema/Screens/Screen[@key=$screens]/Grid/Specific" />
                                        <xsl:for-each select="$categories">
                                            <xsl:variable name="categoryKey" select="." />
                                            <xsl:variable name="countItems" select="count($items[Classification[@key=$classification]=$categoryKey])" />
                                            <xsl:variable name="countSpecificFields" select="count($specifics[@classification=$classification and @category=$categoryKey]/Section/Column-Field)" />
                                            <Count><xsl:value-of select="$countItems * (count($commonFields) + $countSpecificFields)" /></Count>
                                        </xsl:for-each>
                                    </xsl:variable>
                                    <xsl:value-of select="sum($expectedFields/Count)"/>
                                </xsl:variable>                                
                                <xsl:choose>
                                    <xsl:when test="$countExpectedFields > 0">
                                        <xsl:value-of select="$countCompletedFields div $countExpectedFields"/><xsl:text></xsl:text>
                                    </xsl:when>
                                    <xsl:otherwise>-</xsl:otherwise>
                                </xsl:choose>
                            </xsl:variable>
                            <Cell-Number line="{$position}" column="21"><xsl:value-of select="$completeness"/></Cell-Number> 
                            <!-- TTM -->
                            <xsl:variable name="ttm">
                                <xsl:variable name="times">
                                    <xsl:for-each select="$items[exists(@sent)]">
                                        <Days><xsl:value-of select="days-from-duration(xs:dateTime(@sent) - xs:dateTime(@created))" /></Days>
                                    </xsl:for-each>
                                </xsl:variable>
                                <xsl:value-of select="round(avg($times/Days),0)"/>
                            </xsl:variable>                           
                            <Cell-Number line="{$position}" column="22"><xsl:value-of select="$ttm"/></Cell-Number>  
                            <Cell-Number line="{$position}" column="23"><xsl:value-of select="count($items[exists(@sent)])"/></Cell-Number>  
                            <Cell-Number line="{$position}" column="24"><xsl:value-of select="count($items[@updated > @sent])"/></Cell-Number>  
                            <Cell-Number line="{$position}" column="25"><xsl:value-of select="count($items[not(exists(@sent))])"/></Cell-Number>
                        </xsl:variable>

                        <!-- Sheets -->
                        <Sheet>
                            <Sheet-Name><xsl:value-of select="Name"/> - Categories</Sheet-Name>
                            <Cells>
                                <xsl:if test="exists($tabCategories/Cell-Text[@column=4][starts-with(.,'Error:')])">
                                    <Cell-Text column="6" line="3"><xsl:value-of select="count($tabCategories/Cell-Text[@column=4][starts-with(.,'Error:')])"/> broken link(s). View detail below.</Cell-Text>
                                </xsl:if>
                                <Cell-Text line="3" column="3"><xsl:value-of select="Name"/> - Categories</Cell-Text>
                                <Cell-Text line="6" column="3"><xsl:value-of select="Name"/></Cell-Text>
                                <Cell-Text line="6" column="4"><xsl:value-of select="$settings/Supplier/Name"/></Cell-Text>
                                <xsl:copy-of select="$tabCategories"/>
                            </Cells>
                        </Sheet>  
                        <Sheet>
                            <Sheet-Name><xsl:value-of select="Name"/> - Properties</Sheet-Name>
                            <Cells>
                                <xsl:if test="exists($tabProperties/Cell-Text[@column=10][.='Broken'])">
                                    <Cell-Text column="6" line="3"><xsl:value-of select="count($tabProperties/Cell-Text[@column=10][.='Broken'])"/> broken link(s). View detail below.</Cell-Text>
                                </xsl:if>
                                <Cell-Text line="3" column="3"><xsl:value-of select="Name"/> - Properties</Cell-Text>
                                <Cell-Text line="6" column="4"><xsl:value-of select="Name"/></Cell-Text>
                                <Cell-Text line="6" column="5"><xsl:value-of select="$settings/Supplier/Name"/></Cell-Text>
                                <Cell-Text line="6" column="6"><xsl:value-of select="Name"/></Cell-Text>
                                <Cell-Text line="6" column="11"><xsl:value-of select="$settings/Supplier/Name"/></Cell-Text>
                                <xsl:copy-of select="$tabProperties"/>
                            </Cells>
                        </Sheet>  
                        <Sheet>
                            <Sheet-Name><xsl:value-of select="Name"/> - Options</Sheet-Name>
                            <Cells>
                                <xsl:if test="exists($tabOptions/Cell-Text[@column=7][.='Broken'])">
                                    <Cell-Text column="5" line="3"><xsl:value-of select="count($tabOptions/Cell-Text[@column=7][.='Broken'])"/> broken link(s). View detail below.</Cell-Text>
                                </xsl:if>
                                <Cell-Text line="3" column="3"><xsl:value-of select="Name"/> - Options</Cell-Text>
                                <Cell-Text line="6" column="3"><xsl:value-of select="Name"/></Cell-Text>
                                <Cell-Text line="6" column="4"><xsl:value-of select="$settings/Supplier/Name"/></Cell-Text>
                                <Cell-Text line="6" column="5"><xsl:value-of select="Name"/></Cell-Text>
                                <Cell-Text line="6" column="6"><xsl:value-of select="$settings/Supplier/Name"/></Cell-Text>
                                <xsl:copy-of select="$tabOptions"/>
                            </Cells>
                        </Sheet>
                        <Sheet>
                            <Sheet-Name>Main</Sheet-Name>
                            <Cells>
                                <xsl:copy-of select="$tabMain"/>
                                <xsl:choose>
                                    <xsl:when test="exists($tabCategories/Cell-Text[@column=4][starts-with(.,'Error:')]) or exists($tabProperties/Cell-Text[@column=10][.='Broken'])">                        
                                        <Cell-Text column="17" line="{position()+6}">Broken</Cell-Text></xsl:when>
                                    <xsl:otherwise>
                                        <Cell-Text column="17" line="{position()+6}">Valid</Cell-Text>
                                    </xsl:otherwise>
                                </xsl:choose>
                            </Cells> 
                        </Sheet>        
                    </xsl:for-each>
                </Sheets>
            </File>
        </Generate-Excel>
    </xsl:template>
</xsl:stylesheet>