<xsl:stylesheet xmlns:xsl="http://www.w3.org/1999/XSL/Transform"
                xmlns:fn="fn"
                xmlns:pl="http://product-live.com"
                xmlns:xs="http://www.w3.org/2001/XMLSchema" version="3.0" exclude-result-prefixes="xs fn pl">
    
    <xsl:output method="xml" indent="yes" encoding="UTF-8" cdata-section-elements="" />
    
    <!--
         Version : 1.0.1
         Input : Items File
         Params : 
         + table_file : Table to use
         + config_excel_file : Excel template, xlsx-to-xml file
         + output_file_type (!REQUIRED!) (string : ["one_file_per_category","one_file"])  
         `      ->  one_file_per_category : Generate one Excel file by unique pair of classification and category in the input items
         `      ->  one_file : Generate one Excel file that contains all fields and items data
         
         + misc_image_quality_pourcent (int): % of the exported image quality (used to reduce weight of the output file)
         + misc_languages : (string list, separated by commas) : Only  useful for multi-language table. Languages to use, PL codes of languages separated by commas. Output one file per language. 
         + misc_classificationKey : (string, list comma separated) Classifications keys to use, by default we'll use all available classifications in the table
         + misc_display_suffixes (boolean - yes/no) : Base yes, Include suffixes with the displayed values in the Excel cells
         + misc_ignored_field_keys (string list, semicolon separated) : Keys of the fields to ignore in the export, attributes keys separated by semicolon. Exemple : "ATTR1;ATTR2"
         + misc_ignored_field_contains_keys (string list, semicolon separated) : Ignore in the export all the fields whose keys contains those values. Used to ignore all the attributes keys that contains a specific value.
    -->
    <xsl:param name="table_file" select="'TESTS/table.xml'" />
    <xsl:param name="config_excel_file" select="'./TESTS/template.xml'" />
    <xsl:param name="output_file_type" select="'one_file_per_category'" />
    <xsl:param name="misc_image_quality_pourcent" select="45" />
    <xsl:param name="misc_classificationKey" select="''" />
    <xsl:param name="misc_languages" select="''" />
    <xsl:param name="misc_display_suffixes" select="'yes'" />
    <xsl:param name="misc_ignored_field_keys" select="''" />
    <xsl:param name="misc_ignored_field_contains_keys" select="''" />
    
    <xsl:variable name="tableSchema" select="doc($table_file)/Table/Schema" />
    <xsl:variable name="tableIdentifiers" select="$tableSchema/Identifiers/Identifier" />
    <xsl:variable name="tableFields" select="$tableSchema/Fields/Field" />
    <xsl:variable name="tableClassifications" select="$tableSchema/Classifications/Classification" />
    <xsl:variable name="tableElements" select="$tableIdentifiers | $tableFields | $tableClassifications" />
    <xsl:variable name="tableMatrix" select="$tableSchema/Matrix" />
    
    
    <xsl:variable name="config_excel_products_column_index_start" select="3" />
    <xsl:variable name="config_excel_rows" select="document($config_excel_file)/Xlsx-To-Xml/File/Sheets/Sheet[1]/Rows/R" />
    <xsl:variable name="config_excel_common_image_row_index" select="4" />
    <xsl:variable name="config_excel_common_fields_row_index_start" select="5" />
    <xsl:variable name="config_excel_specifics_title" select="'Caractéristiques techniques'" />
    
    <xsl:variable name="misc_display_suffixes_bool" select="normalize-space(upper-case($misc_display_suffixes)) = 'YES'" />
    
    
    <xsl:template match="/">
        <xsl:variable name="allItems" select="Table/Items/Item" />
        
        <xsl:variable name="specific_title_row_index" select="$config_excel_rows[C[@j='2'] = $config_excel_specifics_title]/@i" />
        <xsl:variable name="config_excel_common_fields_row_index_stop" select="$specific_title_row_index - 2" />
        <xsl:variable name="config_excel_specifics_row_index_start" select="$specific_title_row_index + 2" />
        
        <!--Languages-->
        <xsl:variable name="_langs">
            <xsl:choose>
                <xsl:when test="normalize-space($misc_languages) != '' and normalize-space($misc_languages) != 'null'">
                    <xsl:for-each select="tokenize($misc_languages, ',')">
                        <lang>
                            <xsl:value-of select="."/>
                        </lang>
                    </xsl:for-each>
                </xsl:when>
                <xsl:otherwise>
                    <lang>default</lang>
                </xsl:otherwise>
            </xsl:choose>
        </xsl:variable>
        <xsl:variable name="langs" select="$_langs/lang" />
        
        <!--Classifications à utiliser-->
        <xsl:variable name="_classifications">
            <xsl:choose>
                <xsl:when test="normalize-space($misc_classificationKey) != ''">
                    <xsl:copy-of select="$tableClassifications/@key = tokenize($misc_classificationKey,';')"/>
                </xsl:when>
                <xsl:otherwise>
                    <xsl:copy-of select="$tableClassifications"/>
                </xsl:otherwise>
            </xsl:choose>
        </xsl:variable>
        <xsl:variable name="classifications" select="$_classifications/Classification" />
        
        <!--Configuration complète des lignes l'export, en fonction du template et par catégorie -->
        <xsl:variable name="template_config">
            <Config>
                <Common>
                    <Rows>
                        <!--Image-->
                        <xsl:variable name="imageField" select="pl:findFieldByKeyOrTitle($config_excel_rows[@i = $config_excel_common_image_row_index]/C[@j= $config_excel_products_column_index_start])" />
                        <xsl:if test="$imageField">
                            <Row index="{$config_excel_common_image_row_index}" image="true" key="{$imageField/@key}">
                                <Field>
                                    <xsl:copy-of select="$imageField"/>
                                </Field>
                            </Row>
                        </xsl:if>
                        <xsl:for-each select="$config_excel_rows[@i &gt;= $config_excel_common_fields_row_index_start and @i &lt;= $config_excel_common_fields_row_index_stop]">
                            <xsl:variable name="rowIndex" select="@i" />
                            <xsl:variable name="fieldTitleOrKey" select="C[@j='2']" />
                            <xsl:variable name="fieldInfos" select="C[@j='1']" />
                            <xsl:variable name="foundField" select="pl:findFieldByKeyOrTitle($fieldTitleOrKey)" />
                            <xsl:choose>
                                <xsl:when test="$foundField">
                                    <Row index="{$rowIndex}" key="{$foundField/@key}">
                                        <xsl:if test="lower-case(normalize-space($fieldInfos)) = 'tri' or lower-case(normalize-space($fieldInfos)) = 'sort'">
                                            <xsl:attribute name="sort">true</xsl:attribute>
                                        </xsl:if>
                                        <Field>
                                            <xsl:copy-of select="$foundField"/>
                                        </Field>
                                    </Row>
                                </xsl:when>
                                <xsl:otherwise>
                                    <xsl:message terminate="no">
                                        notfound : <xsl:value-of select="$fieldTitleOrKey"/>
                                    </xsl:message>
                                    <Row index="{$rowIndex}" remove="true"/>
                                </xsl:otherwise>
                            </xsl:choose>
                        </xsl:for-each>
                    </Rows>
                </Common>
                <Specifics>
                    <Categories>
                        <xsl:for-each-group select="$allItems[$classifications/@key = Classification/@key]" group-by="concat(Classification/@key,'##', Classification)">
                            <xsl:variable name="classificationKey" select="tokenize(current-grouping-key(),'##')[1]" />
                            <xsl:variable name="categoryKey" select="tokenize(current-grouping-key(),'##')[2]" />
                            <xsl:variable name="categoryItemsDistinctFieldsKeys" select="distinct-values(current-group()/node()/@key)" />
                            <xsl:variable name="category" select="$classifications[@key=$classificationKey]/Categories/Category[@key = $categoryKey]" />
                            <Category key="{$categoryKey}" classification="{$classificationKey}">
                                <xsl:copy-of select="$category/node()[starts-with(name(),'Title')]"/>
                                <Rows>
                                    <xsl:variable name="rows">
                                        <xsl:merge>
                                            <xsl:merge-source name="matrixField" select="$tableMatrix/Specific[@classification = $classificationKey and @category=$categoryKey]/Field[@key = $categoryItemsDistinctFieldsKeys]" sort-before-merge="yes">
                                                <xsl:merge-key select="@key"/>
                                            </xsl:merge-source>
                                            <!--Filtre sur des attributs de data uniquement, et ignore certains attributs -->
                                            <xsl:merge-source name="tableField" select="$tableFields[not(@type = ('ATTACHMENT','IMAGE')) 
                                                    and not(@key = tokenize($misc_ignored_field_keys,';')) 
                                                    and ($misc_ignored_field_contains_keys = '' or ($misc_ignored_field_contains_keys != '' and not(matches(@key,replace($misc_ignored_field_contains_keys,';','|')))))]" sort-before-merge="yes">
                                                <xsl:merge-key select="@key"/>
                                            </xsl:merge-source>
                                            <xsl:merge-action>
                                                <xsl:variable name="tableField" select="current-merge-group('tableField')" />
                                                <xsl:variable name="matrixField" select="current-merge-group('matrixField')" />
                                                <xsl:if test="$matrixField and $tableField">
                                                    <Row index="{position()}" key="{$tableField/@key}">
                                                        <Field>
                                                            <xsl:copy-of select="$tableField"/>
                                                        </Field>
                                                    </Row>
                                                </xsl:if>
                                            </xsl:merge-action>
                                        </xsl:merge>
                                    </xsl:variable>
                                    <!--Recalcul de l'index des spécifiques-->
                                    <xsl:for-each select="$rows/Row">
                                        <xsl:sort select="@index" order="ascending" data-type="number" />
                                        <Row index="{$config_excel_specifics_row_index_start - 1 + position()}">
                                            <xsl:copy-of select="@*[name() != 'index']"/>
                                            <xsl:copy-of select="node()"/>
                                        </Row>
                                    </xsl:for-each>
                                </Rows>
                            </Category>
                        </xsl:for-each-group>
                    </Categories>
                </Specifics>
            </Config>
        </xsl:variable>
        <xsl:variable name="config_common_rows" select="$template_config/Config/Common/Rows/Row" />
        
        <!--Type de tri dans le book de gamme-->
        <xsl:variable name="item_filtering">
            <xsl:choose>
                <xsl:when test="$config_common_rows[@sort][1]">
                    <xsl:variable name="field" select="$config_common_rows[@sort]/Field/node()" />
                    <sort key="{$field/@key}" type="{pl:getSortByFieldType($field/@type)}"/>
                </xsl:when>
                <xsl:otherwise>
                    <xsl:for-each select="$config_common_rows[not(@remove)][field/node()/@type != 'IMAGE']">
                        <xsl:sort select="@index" order="ascending" data-type="number" />
                        <xsl:if test="position() = 1">
                            <xsl:variable name="field" select="field/node()" />
                            <sort key="{$field/@key}" type="{pl:getSortByFieldType($field/@type)}"/>
                        </xsl:if>
                    </xsl:for-each>
                </xsl:otherwise>
            </xsl:choose>
        </xsl:variable>
        
        <!--Configuration complètes des fichiers à générer-->
        <xsl:variable name="files">
            <xsl:for-each select="$langs">
                <xsl:variable name="lang" select="." />
                <xsl:variable name="currentDateTime" select="format-dateTime(pl:to_french_date_time(current-dateTime()), '[M01]-[D01]-[Y0001]_[H01]h[m01]m[s01]s')" />
                <xsl:variable name="documentFileName">
                    <xsl:choose>
                        <xsl:when test="upper-case($lang) = ('FRA','DEFAULT')">Book_de_gamme</xsl:when>
                        <xsl:otherwise>Assortment_plan</xsl:otherwise>
                    </xsl:choose>
                </xsl:variable>
                <xsl:choose>
                    <xsl:when test="$output_file_type = 'one_file'">
                        <File lang="{$lang}">
                            <File-Name>
                                <xsl:value-of select="string-join(($currentDateTime,'_',$documentFileName,'_full','.xlsx'),'')"/>
                            </File-Name>
                            <Items>
                                <xsl:copy-of select="$allItems"/>
                            </Items>
                            <Rows>
                                <xsl:variable name="rows">
                                    <xsl:for-each-group select="$template_config/Config/Specifics/Categories/Category/Rows/Row" group-by="@key">
                                        <xsl:copy-of select="current-group()[1]"/>
                                    </xsl:for-each-group>
                                </xsl:variable>
                                <xsl:copy-of select="$config_common_rows"/>
                                <xsl:for-each select="$rows/Row">
                                    <xsl:sort select="@index" order="ascending" data-type="number" />
                                    <Row index="{$config_excel_specifics_row_index_start - 1 + position()}">
                                        <xsl:copy-of select="@*[name() != 'index']"/>
                                        <xsl:copy-of select="node()"/>
                                    </Row>
                                </xsl:for-each>
                            </Rows>
                        </File>
                    </xsl:when>
                    <xsl:otherwise>
                        <xsl:for-each select="$template_config/Config/Specifics/Categories/Category">
                            <xsl:variable name="category" select="." />
                            <xsl:variable name="categoryKey" select="$category/@key" />
                            <xsl:variable name="categoryClassificationKey" select="$category/@classification" />
                            <xsl:variable name="categoryTitle" select="pl:get_localized_field_title(.,$lang)" />
                            <File lang="{$lang}">
                                <File-Name>
                                    <xsl:value-of select="string-join(($currentDateTime,'_',$documentFileName,'_',pl:getBanalisedFileName($categoryTitle),'.xlsx'),'')"/>
                                </File-Name>
                                <Items>
                                    <xsl:copy-of select="$allItems[Classification and Classification[@key = $categoryClassificationKey and . = $categoryKey]]"/>
                                </Items>
                                <Rows>
                                    <xsl:copy-of select="$config_common_rows"/>
                                    <xsl:copy-of select="$category/Rows/Row"/>
                                </Rows>
                            </File>
                        </xsl:for-each>
                        <!--Produits sans catégorie-->
                        <xsl:if test="$allItems[not(Classification)]">
                            <File lang="{$lang}">
                                <File-Name>
                                    <xsl:value-of select="string-join(($currentDateTime,'_',$documentFileName,'_common','.xlsx'),'')"/>
                                </File-Name>
                                <Items>
                                    <xsl:copy-of select="$allItems[not(Classification)]"/>
                                </Items>
                                <Rows>
                                    <xsl:copy-of select="$config_common_rows"/>
                                </Rows>
                            </File>
                        </xsl:if>
                    </xsl:otherwise>
                </xsl:choose>
            </xsl:for-each>
        </xsl:variable>
        
        <xsl:message terminate="no">
        <xsl:copy-of select="$item_filtering"/>
        </xsl:message>
        <Generate-Excel>            
            <xsl:for-each select="$files/File">
                <xsl:variable name="file" select="." />
                <xsl:variable name="rows" select="$file/Rows/Row" />
                <xsl:variable name="items" select="$file/Items/Item" />
                <xsl:variable name="lang" select="@lang" />
                <File>
                    <xsl:copy-of select="File-Name"/>
                    <Template-Key>template</Template-Key>
                    <Sheets>
                        <Sheet>
                            <Sheet-Name>Data</Sheet-Name>
                            <Cells>
                                <!--Clean des textes de config restant-->
                                <xsl:for-each select="$config_common_rows">
                                    <Cell-Text line="{@index}" column="1">
                                        <xsl:value-of select="' '"/>
                                    </Cell-Text>
                                </xsl:for-each>
                                <!--Vide les textes d'aide ou reste de config-->
                                <Cell-Text line="{$config_excel_common_fields_row_index_stop + 1}" column="2">
                                    <xsl:value-of select="' '"/>
                                </Cell-Text>
                                <Cell-Text line="{$config_excel_common_image_row_index}" column="{$config_excel_products_column_index_start}">
                                    <xsl:value-of select="' '"/>
                                </Cell-Text>
                                
                                <!--Titre des attributs spécifiques-->
                                <Cell-Text line="{$specific_title_row_index}" column="2">
                                    <xsl:choose>
                                        <xsl:when test="upper-case($lang) = ('FRA','DEFAULT')"><xsl:value-of select="$config_excel_specifics_title"/></xsl:when>
                                        <xsl:otherwise>Technical characteristics</xsl:otherwise>
                                    </xsl:choose>
                                </Cell-Text>
                                
                                <!-- Ajout des libellés des lignes-->
                                <xsl:for-each select="$rows">
                                    <xsl:sort select="@index" order="ascending" data-type="number" />
                                    <xsl:variable name="row" select="." />                         
                                    <Cell-Text line="{@index}" column="2">
                                        <xsl:choose>
                                            <xsl:when test="@remove = 'true' or @image = 'true'">
                                                <xsl:value-of select="'  '"/>
                                            </xsl:when>
                                            <xsl:otherwise>
                                                <xsl:value-of select="pl:get_localized_field_title($row/Field/node(),$lang)"/>
                                            </xsl:otherwise>
                                        </xsl:choose>
                                    </Cell-Text>
                                </xsl:for-each>
                                
                                
                                <!--Items avec données -->
                                <xsl:for-each select="$items">
                                    <xsl:sort select="node()[@key=$item_filtering/sort/@key]" order="ascending" data-type="{$item_filtering/sort/@type}" />
                                    <xsl:variable name="item" select="." />
                                    <xsl:variable name="column" select="($config_excel_products_column_index_start - 1) + position()" />
                                    
                                    <!--Attributs-->
                                    <xsl:merge>
                                        <xsl:merge-source name="row" select="$rows[not(@remove)]" sort-before-merge="yes">
                                            <xsl:merge-key select="@key"/>
                                        </xsl:merge-source>
                                        <xsl:merge-source name="itemData" select="$item/node()" sort-before-merge="yes">
                                            <xsl:merge-key select="@key"/>
                                        </xsl:merge-source>
                                        <xsl:merge-action>
                                            <xsl:variable name="row" select="current-merge-group(('row'))" />
                                            <xsl:variable name="itemData" select="current-merge-group(('itemData'))" />
                                            <xsl:if test="$row and $itemData">
                                                <xsl:variable name="field" select="$row/Field/node()" />
                                                <xsl:copy-of select="pl:createData($itemData,$field, $lang, $row/@index, $column )"/>
                                            </xsl:if>
                                        </xsl:merge-action>
                                    </xsl:merge>
                                </xsl:for-each>
                            </Cells>
                        </Sheet>
                    </Sheets>
                </File>
            </xsl:for-each>
        </Generate-Excel>
    </xsl:template>
    
    
    <xsl:function name="pl:findFieldByKeyOrTitle">
        <xsl:param name="keyOrTitle" />
        <xsl:variable name="byKey" select="$tableElements[@key = $keyOrTitle]" />
        <xsl:choose>
            <xsl:when test="$byKey">
                <xsl:copy-of select="$byKey"/>
            </xsl:when>
            <xsl:otherwise>
                <xsl:copy-of select="$tableElements[normalize-space(Title) = normalize-space($keyOrTitle)]"/>
            </xsl:otherwise>
        </xsl:choose>
    </xsl:function>
    
    <xsl:function name="pl:getSortByFieldType">
        <xsl:param name="fieldType" />
        <xsl:choose>
            <xsl:when test="$fieldType = 'NUMBER'">number</xsl:when>
            <xsl:otherwise>text</xsl:otherwise>
        </xsl:choose>
    </xsl:function>
    
    <!--Version 1.0-->
    <xsl:function name="pl:get_localized_field_title">
        <xsl:param name="element" />
        <xsl:param name="lang" />
        
        <xsl:variable name="baseTitle" select="$element/Title" />
        <xsl:choose>
            <xsl:when test="not($lang) or normalize-space($lang) = '' or normalize-space($lang) = 'default'">
                <xsl:value-of select="$baseTitle"/>
            </xsl:when>
            <xsl:otherwise>
                <xsl:variable name="foundLang" select="$element/Title-Local[upper-case(normalize-space(@lang)) = upper-case(normalize-space($lang))]" />
                <xsl:value-of select="if($foundLang and normalize-space($foundLang) != '') then $foundLang else $baseTitle"/>
            </xsl:otherwise>
        </xsl:choose>
    </xsl:function>
    
    
    <xsl:function name="pl:createData">
        <xsl:param name="fieldValue"/>
        <xsl:param name="tableField" />
        <xsl:param name="lang" />
        <xsl:param name="line"/>
        <xsl:param name="column"/>
        <xsl:choose>
            <xsl:when test="$tableField/name() = 'Identifier'and  $fieldValue">
                <Cell-Text line="{$line}" column="{$column}">
                    <xsl:value-of select="$fieldValue"/>
                </Cell-Text>
            </xsl:when>
            <xsl:when test="$tableField/name() = 'Classification'and  $fieldValue">
                <Cell-Text line="{$line}" column="{$column}">
                    <xsl:variable name="classifKey" select="$fieldValue/@key" />
                    <xsl:variable name="categoryKey" select="$fieldValue" />
                    <xsl:variable name="foundCategory" select="$tableClassifications[@key =$classifKey]/Categories/Category[@key=$categoryKey]" />
                    <xsl:value-of select="if($foundCategory) then pl:get_localized_field_title($foundCategory,$lang) else $fieldValue"/>
                </Cell-Text>
            </xsl:when>
            <xsl:when test="$tableField and $tableField/@type='IMAGE' and $fieldValue">
                <Cell-Image line="{$line}" column="{$column}">
                    <xsl:value-of select="normalize-space(string-join((pl:get_PL_API_link($fieldValue),'/ext:jpg/quality:',$misc_image_quality_pourcent), ''))"/>
                </Cell-Image>
            </xsl:when>
            <xsl:when test="$tableField and $tableField/@type='SINGLE-SELECT' and $fieldValue">
                <xsl:variable name="foundOption" select="$tableField/Options/Option[@key=$fieldValue]" />
                <Cell-Text line="{$line}" column="{$column}">
                    <xsl:value-of select="if($foundOption) then pl:get_localized_field_title($foundOption,$lang) else $fieldValue"/>
                </Cell-Text>
            </xsl:when>
            <xsl:when test="$tableField and $tableField/@type= ('MULTIPLE-SELECT','MULTIPLE-SELECT-QUANTIFIED','MULTIPLE-SELECT-QUANTIFIED-WITH-COMMENTS') and $fieldValue">
                <Cell-Text line="{$line}" column="{$column}">
                    <xsl:for-each select="$fieldValue">
                        <xsl:variable name="key" select="." />
                        <xsl:variable name="foundOption" select="$tableField/Options/Option[@key=$key]" />
                        <xsl:value-of select="if($foundOption) then pl:get_localized_field_title($foundOption,$lang) else $fieldValue"/>
                        <xsl:if test="position()!=last()">
                            <xsl:text> / </xsl:text>
                        </xsl:if>
                    </xsl:for-each>
                </Cell-Text>
            </xsl:when>
            <xsl:when test="$tableField and $tableField/@type='NUMBER' and $fieldValue">
                <Cell-Text line="{$line}" column="{$column}">
                    <xsl:value-of select="$fieldValue"/>
                    <xsl:choose>
                        <xsl:when test="$fieldValue/@suffix and $misc_display_suffixes_bool">
                            <xsl:value-of select="concat(' ',pl:get_localized_field_title($tableField/Suffixes/Suffix[@key=$fieldValue/@suffix], $lang))"/>
                        </xsl:when>
                        <xsl:when test="$tableField/Suffix and $misc_display_suffixes_bool">
                            <xsl:value-of select="concat(' ',$tableField/Suffix)"/>
                        </xsl:when>
                    </xsl:choose>
                </Cell-Text>
            </xsl:when>
            <xsl:when test="$fieldValue">
                <Cell-Text line="{$line}" column="{$column}">
                    <xsl:value-of select="$fieldValue"/>
                </Cell-Text>
            </xsl:when>
        </xsl:choose>
    </xsl:function>
    
    <xsl:function name="pl:to_french_date_time">
        <xsl:param name="date"  as="xs:dateTime"/>
        <xsl:value-of select="xs:dateTime(format-dateTime($date, '[Y0001]-[M01]-[D01]T[H01]:[m01]:[s01].[f]Z', 'en', (), 'Europe/Amsterdam'))"/>
    </xsl:function>
    
    <xsl:function name="pl:getBanalisedFileName">
        <xsl:param name="value" />
        <xsl:value-of select="
            normalize-space(
                replace(
                    replace(
                        replace($value,'\\','_'),
                        '/', '_'),
                    '[^A-Za-z0-9(à,@,é,è):\-_ ]', '')
            )" />
    </xsl:function>
    
    <!--Version 1.3-->
    <xsl:function name="pl:get_PL_API_link">
        <xsl:param name="PLlink" />
        <xsl:if test="normalize-space($PLlink) != ''">
            <xsl:choose>
                <xsl:when test="contains($PLlink, '/download/')">
                    <!--Build base URL-->
                    <xsl:variable name="step_1" select="replace(
                            replace($PLlink,'https://app.product-live.com/data-factory/',
                                'https://asset.product-live.com/documents/'),
                            '/download/','/')" />
                    <!--Remove last slash value (filename usually) -->
                    <xsl:variable name="step_2" select="replace($step_1, tokenize($step_1,'/')[last()],'')" />
                    <!--Finally, remove trailing /-->
                    <xsl:value-of select="substring($step_2, 1, string-length($step_2) - 1)"/>
                </xsl:when>
                <xsl:otherwise>
                    <!--Remove useless values-->
                    <xsl:variable name="step_1" select="
                        replace(
                            replace(
                                replace($PLlink,'https://asset.prod.product-live.com/file-map/',''),  
                                'https://asset.prod.product-live.com/file-map-resize/',''),
                            '_documents/redirect', '')" />
                    <xsl:variable name="ids" select="tokenize($step_1,'/')" />
                    <xsl:value-of select="string-join(('https://asset.product-live.com/documents',normalize-space($ids[1]), normalize-space($ids[2])),'/')"/>
                </xsl:otherwise>
            </xsl:choose>            
        </xsl:if>        
    </xsl:function>
    
    
</xsl:stylesheet>